using System.Linq;
using RimWorld;
using UnityEngine;
using Verse;
using Verse.AI;

namespace Therapy
{
    public class JoyGiver_ReceiveTherapy : JoyGiver
    {
        public override float GetChance(Pawn pawn)
        {
            if (pawn.needs.joy == null) return base.GetChance(pawn) + pawn.GetTherapyNeed();

            if (pawn.needs.joy.CurLevelPercentage < 0.1f) return base.GetChance(pawn);
            return Mathf.Max(0, base.GetChance(pawn) + pawn.GetTherapyNeed() - (1 - pawn.needs.joy.CurLevelPercentage) * 6);
        }

        public override Job TryGiveJob(Pawn pawn)
        {
            if (!InteractionUtility.CanInitiateInteraction(pawn))
            {
                //Log.Warning(pawn.NameStringShort + ": can't initiate interaction");
                return null;
            }
            var couch = FindRandomCouchWithChair(pawn);
            if (couch == null)
            {
                //Log.Warning(pawn.NameStringShort + ": no couch with chair");
                return null;
            }
            return new Job(def.jobDef, couch);
        }

        public static Building_Couch FindRandomCouchWithChair(Pawn pawn)
        {
            var buildingCouches = pawn.Map.listerBuildings.AllBuildingsColonistOfClass<Building_Couch>();
            var couches = buildingCouches.Where(x => CanUse(pawn, x));
            Building_Couch result;
            if (!couches.TryRandomElementByWeight(couch => VisitChanceScore(pawn, couch), out result))
            {
                return couches.FirstOrDefault();
            }
            return result;
        }

        private static float VisitChanceScore(Pawn pawn, Building_Couch couch)
        {
            var room = couch.GetRoom();
            if (room == null) return 0;

            //float num = GenMath.LerpDouble(-100f, 100f, 0.05f, 2f, pawn.relations.OpinionOf(therapist));
            float lengthHorizontal = (pawn.Position - couch.Position).LengthHorizontal;
            float num2 = Mathf.Clamp(GenMath.LerpDouble(0f, 150f, 1f, 0.2f, lengthHorizontal), 0.2f, 1f);
            float num3 = Mathf.Max(0.1f, room.GetStat(RoomStatDefOf.Beauty));
            return num2 * num3;
        }

        public static bool CanUse(Pawn pawn, Building_Couch couch)
        {
            var canReserveAndReach = pawn.CanReserveAndReach(couch, PathEndMode.InteractionCell, Danger.None) && !couch.IsForbidden(pawn);
            Thing chair;
            var hasChair = MainUtility.TryFindChairNearCouch(couch, out chair);
            //Log.Message(couch.Label + " at " + couch.Position 
            //    + (hasChair ? " has chair" : " has no chair") 
            //    + (couch.IsForbidden(pawn) ? " and is forbidden" : " and is not forbidden")
            //    + (pawn.CanReserveAndReach(couch, PathEndMode.InteractionCell, Danger.None) ? " and can be reached" : " and can't be reached"));
            return canReserveAndReach && hasChair;
        }
    }
}
